# Casio dictionary NAND Update Image builder - structs.py
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.


import struct, ctypes
from datetime import datetime

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

def dec_to_bcd(num):
   bcd = 0
   shift = 0
   while (num != 0):
     digit = num % 10
     bcd |= digit << shift
     shift += 4
     num = num // 10
   return bcd

def read_struct(f, struct):
    s = struct()
    slen = ctypes.sizeof(s)
    if isinstance(f, bytes):
      byte_data = f
    else:
      byte_data = f.read(slen)
    fit = min(len(byte_data), slen)
    ctypes.memmove(ctypes.addressof(s), byte_data, fit)
    return s

class CheckSum16(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint16_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class CheckSum32(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint32_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)
      

class PVDateTime(ctypes.BigEndianStructure):
    _fields_ = [ ("bytes", uint8_t*8) ]
    def __str__(self):
        vals = tuple(self.bytes[i]&0xFF for i in range(7))
        return "%02X%02X/%02X/%02X %02X:%02X:%02X" % vals

    def pformat(self):
        return [self.__str__()]

    @staticmethod
    def from_string(dt_str):
      dt = datetime.strptime(dt_str, "%Y/%m/%d %H:%M:%S")
      obj = PVDateTime()
      obj.bytes[0] = (dec_to_bcd(dt.year) >> 8) & 0xff
      obj.bytes[1] = dec_to_bcd(dt.year) & 0xff
      obj.bytes[2] = dec_to_bcd(dt.month) & 0xff
      obj.bytes[3] = dec_to_bcd(dt.day) & 0xff
      obj.bytes[4] = dec_to_bcd(dt.hour) & 0xff
      obj.bytes[5] = dec_to_bcd(dt.minute) & 0xff
      obj.bytes[6] = dec_to_bcd(dt.second) & 0xff
      obj.bytes[7] = 0x00
      return obj

    @staticmethod
    def now():
      dt = datetime.now()
      obj = PVDateTime()
      obj.bytes[0] = (dec_to_bcd(dt.year) >> 8) & 0xff
      obj.bytes[1] = dec_to_bcd(dt.year) & 0xff
      obj.bytes[2] = dec_to_bcd(dt.month) & 0xff
      obj.bytes[3] = dec_to_bcd(dt.day) & 0xff
      obj.bytes[4] = dec_to_bcd(dt.hour) & 0xff
      obj.bytes[5] = dec_to_bcd(dt.minute) & 0xff
      obj.bytes[6] = dec_to_bcd(dt.second) & 0xff
      obj.bytes[7] = 0x00
      return obj

class ExtModelStr(ctypes.BigEndianStructure):
    _fields_ = [ ("chars", char*46) ]
    def __str__(self):
        slen = self.chars[0]
        if slen == 0xFF:
          return ""
        return (self.chars[1:slen+1])
    def pformat(self):
        return [self.__str__()]

class DicsPreloadFile(ctypes.BigEndianStructure):
  _fields_ = [
    ("filename",  char*28),    # name
    ("idx",       uint32_t),   #
  ]        

class DicsFareaHeader(ctypes.BigEndianStructure):
  _fields_ = [
    ("count",  uint16_t),    # number of preload files
    ("filler", char*14),  #
  ]

class DirEntry(ctypes.BigEndianStructure):
  _fields_ = [
    ("name",      char*16),
    ("dir_id",    uint16_t),
    ("parent_id", uint16_t),
    ("location",  uint32_t),
    ("size",      uint32_t),
    ("flags",     uint32_t),
  ]

class PVOSHeader1(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIOPVOS200
    ("magic",     uint16_t),  # 0x55AA
    ("blocksize", uint16_t),  # 0x200
    ("flash_nblks", uint32_t), #
    ("update_nblks",uint32_t), #
    ("dword18",   uint32_t), #  Zero?
    ("hdr1_2_blk",uint32_t), # blk no of second copy of hdr1??
    ("dir_blk",   uint32_t), # start blk of directory
    ("dir_nblks", uint32_t), # blk length of directory
    ("hdr2_blk",  uint32_t), # blk of second hdr
    ("hdr1_2_cpy",uint32_t), # blk no of second copy of hdr1?? always the same as hdr1_2_blk
    ("dword30",   uint32_t), # always 0xffffffff
    ("dword34",   uint32_t), # always 0xffffffff
    ("dword38",   uint32_t), #
    ("dword3C",   uint32_t), #
    ("dword40",   uint32_t), # always 0x0?
    ("dword44",   uint32_t), #
    ("model",     char * 4), # 
    ("dword4C",   uint32_t), #
    ("img_nblks", uint32_t), # length of OS update image file in blocks
    ("dword54",   uint32_t), #
    ("dword58",   uint32_t), #
    ("dword5C",   uint32_t), #
    ("dword60",   uint32_t), #
    ("dword64",   uint32_t), #
    ("ext_model", char*8),   # extended model string
    ("filler70",  char*398), #
    ("checksum",  CheckSum16), #
   ]
   def pack(self):
     self.checksum.init()
     self.checksum.update(bytes(self)[:-2])
     return bytes(self)[:]

class PVOSHeader2(ctypes.BigEndianStructure):
  _fields_ = [
    ("signature", char * 12), # CASIOPVOS???
    ("dwordC",    uint32_t), # All Zeros
    ("dword10",   uint32_t), # All Zeros
    ("dword14",   uint32_t), # All FF
    ("model",     char * 4), # model string
    ("datetime",  PVDateTime), # date time (bcd encoded)
    ("version",   uint16_t), # version number (bcd encoded)
    ("pad26",     uint16_t), # padding should be zero
    ("dword28",   uint32_t), # All Zeros
    ("datachksum",CheckSum32), # NAND Checksum
    ("ext_model", char * 8), # ext model string
    ("filler34",  char * 454), #
    ("checksum",  CheckSum16), #
  ]
  def pack(self, timestamp=None):
    if timestamp:
        self.datetime = PVDateTime.from_string(timestamp)
    else:
        self.datetime = PVDateTime.now()
    self.checksum.init()
    self.checksum.update(bytes(self)[:-2])
    return bytes(self)[:]
    
class CasioDicHeaderR(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03R
    ("model",     char * 4),  # L865
    ("magic",     uint32_t),  # 0x55AAAA55
    ("flash_size",uint32_t),  # total system flash size?
    ("block_size",uint32_t),  # flash block size?
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("dword20",   uint32_t),  #
    ("b_offset",  uint32_t),  #
    ("dword28",   uint32_t),  #
    ("dword2C",   uint32_t),  #
    ("boot_cksum",CheckSum32),# checksum of start data until this header
    ("extmodel",  char * 8),  # CY101
    ("filler3C",  uint8_t*66), #
    ("checksum",  CheckSum16),  # header checksum
   ]
   def pack(self):
     self.checksum.init()
     self.checksum.update(bytes(self)[:-2])
     return bytes(self)[:]
     
class CasioDicHeaderB(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("a_offset",  uint32_t),  # offset of the A header
    ("entrypoint",uint32_t),  # main code entry point
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("farea_off", uint32_t),  #
    ("farea_len", uint32_t),  #
    ("datetime1", PVDateTime),#
    ("datetime2", PVDateTime),#
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("datasum1",  CheckSum32),  #
    ("datasum2",  CheckSum32),  #
    ("rom_cksum", CheckSum32),  #
    ("filler44",  uint8_t*12),#
    ("extmodel",  ExtModelStr), # CY101
    ("checksum",  CheckSum16),  # header checksum
   ]
   def pack(self):
     self.checksum.init()
     self.checksum.update(bytes(self)[:-2])
     return bytes(self)[:]
