# Casio dictionary NAND Update Image parser - nand_parse.py
# This script can parse a .bin file generated by nand_build.py,
# extract all files, and show generation parameters.
# It can also detect the layout .ini file used to build the image.

import sys
import os
import ctypes
import argparse
import glob
import configparser
from structs import PVOSHeader1, PVOSHeader2, DirEntry, read_struct, CheckSum16, CheckSum32

class Logger:
    """A simple logger to write to both file and console."""
    def __init__(self, filepath):
        self.terminal = sys.stdout
        self.logfile = open(filepath, 'w', encoding='utf-8')

    def write(self, message):
        self.terminal.write(message)
        self.logfile.write(message)
        self.logfile.flush()

    def close(self):
        self.logfile.close()

    def __getattr__(self, attr):
        # Make it compatible with file-like objects
        return getattr(self.terminal, attr)

# --- Checksum Verification ---
def verify_checksums(bin_file):
    """
    Verifies the checksums of a Casio NAND image .bin file.
    """
    print(f"--- Verifying Checksums for {os.path.basename(bin_file)} ---")
    try:
        with open(bin_file, 'rb') as f:
            # Read headers to get necessary info
            f.seek(0)
            hdr1 = read_struct(f, PVOSHeader1)
            
            if hdr1.blocksize == 0:
                print("Error: Block size is zero, cannot proceed.")
                return
                
            f.seek(hdr1.hdr2_blk * hdr1.blocksize)
            hdr2 = read_struct(f, PVOSHeader2)

            # Calculate Header 1 checksum
            f.seek(0)
            hdr1_data = f.read(ctypes.sizeof(PVOSHeader1))
            calc_hdr1_sum = CheckSum16()
            calc_hdr1_sum.update(hdr1_data[:-2])

            # Calculate Header 2 checksum
            f.seek(hdr1.hdr2_blk * hdr1.blocksize)
            hdr2_data = f.read(ctypes.sizeof(PVOSHeader2))
            calc_hdr2_sum = CheckSum16()
            calc_hdr2_sum.update(hdr2_data[:-2])

            # Calculate NAND data checksum
            calc_nand_sum = CheckSum32()
            f.seek(0, 2)
            total_blocks = f.tell() // hdr1.blocksize
            
            data_start_block = hdr1.hdr2_blk + 1
            f.seek(data_start_block * hdr1.blocksize)
            
            num_data_blocks = total_blocks - data_start_block
            for _ in range(num_data_blocks):
                data_chunk = f.read(hdr1.blocksize)
                calc_nand_sum.update(data_chunk)

            # Print results
            print(f"Model: {hdr1.model.decode('ascii', errors='ignore').strip()}")
            print(f"Timestamp: {hdr2.datetime}")
            print("--- PVOS Header 1 ---")
            stored_hdr1_sum = hdr1.checksum.chksum
            print(f"hdr_1.checksum: {stored_hdr1_sum:x} ({calc_hdr1_sum.chksum:x}) {'OK' if stored_hdr1_sum == calc_hdr1_sum.chksum else 'Fail'}")
            
            print("--- PVOS Header 2 ---")
            stored_nand_sum = hdr2.datachksum.chksum
            stored_hdr2_sum = hdr2.checksum.chksum
            print(f"hdr_2.datachksum: {stored_nand_sum:x} ({calc_nand_sum.chksum:x}) {'OK' if stored_nand_sum == calc_nand_sum.chksum else 'Fail'}")
            print(f"hdr_2.checksum: {stored_hdr2_sum:x} ({calc_hdr2_sum.chksum:x}) {'OK' if stored_hdr2_sum == calc_hdr2_sum.chksum else 'Fail'}")

    except FileNotFoundError:
        print(f"Error: File not found at '{bin_file}'")
    except Exception as e:
        print(f"An error occurred during checksum verification: {e}")


# --- Layout Detection Feature ---
def load_layouts_from_ini(layouts_dir):
    """
    Dynamically loads layout information from .ini files in a given directory.
    """
    layouts = []
    ini_files = glob.glob(os.path.join(layouts_dir, '*.ini'))
    if not ini_files:
        return []

    for ini_path in ini_files:
        try:
            config = configparser.ConfigParser()
            config.read(ini_path)
            section = config.sections()[0]
            
            layout_info = {
                "name": os.path.basename(ini_path),
                "signature": config.get(section, 'signature').encode('ascii'),
                "dir_blk": int(config.get(section, 'dir_blk'), 16),
                "dir_nblks": int(config.get(section, 'dir_nblks'), 16)
            }
            layouts.append(layout_info)
        except Exception as e:
            pass
    
    return layouts

def detect_layout(bin_file, layouts_dir):
    """
    Detects which layout .ini file was used by comparing header fields
    against dynamically loaded .ini files.
    """
    known_layouts = load_layouts_from_ini(layouts_dir)
    if not known_layouts:
        return "Unknown (could not load any layout configs)"

    try:
        with open(bin_file, 'rb') as f:
            hdr1_data = f.read(ctypes.sizeof(PVOSHeader1))
            if len(hdr1_data) < ctypes.sizeof(PVOSHeader1):
                return "Unknown (file too small)"
            
            hdr1 = read_struct(hdr1_data, PVOSHeader1)
            signature = hdr1.signature.partition(b'\x00')[0]

            for layout in known_layouts:
                if (
                    signature == layout["signature"] and
                    hdr1.dir_blk == layout["dir_blk"] and
                    hdr1.dir_nblks == layout["dir_nblks"]):
                    return layout["name"]
            
            return "Unknown (no matching layout found)"
    except FileNotFoundError:
        return f"Error: File not found at '{bin_file}'"
    except Exception as e:
        return f"An error occurred: {e}"

# --- Main Parsing Feature ---
def parse_nand(bin_file, output_dir, manifest_path):
    """
    Parses a Casio NAND image .bin file, extracts its contents,
    and creates a log and a build manifest.
    """
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
        print(f"Created output directory: {output_dir}")

    def log_struct(struct_obj, name):
        print(f"--- {name} ---")
        for field_name, _ in struct_obj._fields_:
            value = getattr(struct_obj, field_name)
            if hasattr(value, 'chksum'):
                value_str = f"0x{value.chksum:x}"
            elif isinstance(value, bytes):
                value_str = value.partition(b'\x00')[0].decode('ascii', errors='ignore')
            elif isinstance(value, int):
                value_str = f"{value} (0x{value:x})"
            else:
                value_str = str(value)
            print(f"  {field_name}: {value_str}")
        print("-" * (len(name) + 6))

    with open(manifest_path, 'w', encoding='utf-8') as manifest_f, \
         open(bin_file, 'rb') as f:
        
        hdr1 = read_struct(f, PVOSHeader1)
        f.seek(hdr1.hdr2_blk * hdr1.blocksize)
        hdr2 = read_struct(f, PVOSHeader2)

        log_struct(hdr1, "Header 1 Info")
        log_struct(hdr2, "Header 2 Info")

        dir_entries_start_offset = (hdr1.dir_blk + 1) * hdr1.blocksize
        dir_entries_size = (hdr1.dir_nblks - 1) * hdr1.blocksize
        f.seek(dir_entries_start_offset)
        dir_data = f.read(dir_entries_size)

        dir_entries = []
        offset = 0
        sizeof_direntry = ctypes.sizeof(DirEntry)
        while offset < len(dir_data):
            entry_data = dir_data[offset:offset+sizeof_direntry]
            if len(entry_data) < sizeof_direntry: break
            entry = read_struct(entry_data, DirEntry)
            if not entry.name or entry.name[0] in (0xff, 0x00): break
            dir_entries.append(entry)
            offset += sizeof_direntry

        print(f"\nFound {len(dir_entries)} directory entries.")
        print("--- Directory Entries (in original order) ---")
        for i, entry in enumerate(dir_entries):
            print(f"  Entry {i}:")
            print(f"    name: {entry.name.partition(b'\\x00')[0].decode('ascii', 'ignore')}")
            print(f"    dir_id: {entry.dir_id}, parent_id: {entry.parent_id}")
            print(f"    location: {entry.location}, size: {entry.size}, flags: 0x{entry.flags:x}")

        print("\n--- Extracting Files and Building Manifest ---")
        manifest_f.write("# Manifest of files in original order for nand_build.py\n")
        
        dirs = {}
        for entry in dir_entries:
            if entry.parent_id == 0xffff:
                dir_name = entry.name.partition(b'\x00')[0].decode('ascii', errors='ignore')
                dir_path = os.path.join(output_dir, dir_name)
                if not os.path.exists(dir_path):
                    os.makedirs(dir_path)
                dirs[entry.dir_id] = dir_name
                print(f"Created directory: {dir_name}")
                manifest_f.write(f"d {dir_name}\n")

        for entry in dir_entries:
            if entry.parent_id != 0xffff:
                if entry.parent_id in dirs:
                    parent_dir_name = dirs[entry.parent_id]
                    file_name = entry.name.partition(b'\x00')[0].decode('ascii', errors='ignore')
                    rel_path = os.path.join(parent_dir_name, file_name)
                    output_path = os.path.join(output_dir, rel_path)
                    
                    print(f"Extracting: {rel_path} (size: {entry.size} bytes)")
                    manifest_f.write(f"f {rel_path}\n")
                    
                    f.seek(entry.location * hdr1.blocksize)
                    file_data = f.read(entry.size)
                    with open(output_path, 'wb') as out_f:
                        out_f.write(file_data)
                else:
                    file_name_str = entry.name.partition(b'\x00')[0].decode('ascii', 'ignore')
                    print(f"Warning: File '{file_name_str}' has unknown parent_id {entry.parent_id} and will be skipped.")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Parse a Casio NAND BIN file, detect its source layout, or verify checksums.",
        formatter_class=argparse.RawTextHelpFormatter
    )
    parser.add_argument("input_bin", help="Path to the input .bin file.")
    parser.add_argument(
        "output_dir",
        nargs='?',
        default=None,
        help="Directory to extract files into. Required for parsing, ignored for detection or verification."
    )
    parser.add_argument("--detect", action="store_true", help="Only detect the layout .ini file and exit.")
    parser.add_argument("--verify", action="store_true", help="Only verify the NAND image checksums and exit.")

    args = parser.parse_args()

    if not os.path.exists(args.input_bin):
        print(f"Error: Input file not found at '{args.input_bin}'")
        sys.exit(1)

    # Determine base name for log/manifest files
    if args.output_dir:
        base_name = os.path.basename(os.path.normpath(args.output_dir))
    else:
        base_name = os.path.splitext(os.path.basename(args.input_bin))[0]

    log_path = f"{base_name}_info.log"
    logger = Logger(log_path)
    original_stdout = sys.stdout
    sys.stdout = logger

    script_dir = os.path.dirname(os.path.realpath(__file__))
    layouts_path = os.path.join(script_dir, '_layouts')

    try:
        if args.verify:
            verify_checksums(args.input_bin)
            sys.stdout = original_stdout
            logger.close()
            print(f"Verification results saved to: {log_path}")

        elif args.detect:
            layout_name = detect_layout(args.input_bin, layouts_path)
            print(f"Result: The most likely layout is '{layout_name}'")
            sys.stdout = original_stdout
            logger.close()
            print(f"Detection results saved to: {log_path}")

        elif args.output_dir:
            manifest_path = f"{base_name}_manifest.txt"
            
            print("--- Auto-detecting layout before parsing ---")
            layout_name = detect_layout(args.input_bin, layouts_path)
            print(f"Detected Layout: {layout_name}")
            print("-" * 40)

            print("--- Auto-verifying checksums before parsing ---")
            verify_checksums(args.input_bin)
            print("-" * 40)

            print(f"Starting full parse of '{os.path.basename(args.input_bin)}' into '{args.output_dir}'...")
            parse_nand(args.input_bin, args.output_dir, manifest_path)
            
            sys.stdout = original_stdout
            logger.close()
            print(f"\nExtraction complete.")
            print(f"An information log has been saved to: {log_path}")
            print(f"A build manifest has been saved to: {manifest_path}")
        else:
            sys.stdout = original_stdout
            logger.close()
            parser.print_help()

    except Exception as e:
        sys.stdout = original_stdout
        logger.close()
        import traceback
        traceback.print_exc()