# Casio dictionary NAND Update Image builder - nand.py
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.


import os, os.path
from structs import DirEntry
from structs import PVOSHeader1
from structs import PVOSHeader2


class NandGenerator:
  def __init__(self, path, layout, timestamp=None):
    self.entries = []
    self.layout = layout
    self.path = path
    self.timestamp = timestamp
    temp = 0;
    for infile in os.listdir(path):
      if os.path.isdir(os.path.join(path, infile)):
        self.entries.append(DirEntry(name=infile.encode('ascii'), dir_id=temp, parent_id=0xffff, size=0, flags=0x8316))
        temp += 1
    for entry in self.entries[:]:
      entry.location = temp
      for infile in os.listdir(os.path.join(path, entry.name.decode('ascii'))):
        sz = os.path.getsize(os.path.join(path, entry.name.decode('ascii'), infile))
        self.entries.append(DirEntry(name=infile.encode('ascii'), dir_id=0xffff, parent_id=entry.dir_id, size=sz, flags=0x010c))
        temp += 1
    filler_bytes = (self.layout.getint("dir_nblks") - self.dir_used_nblks()) * 512;
    temp = self.sector_align((len(self.entries) * 32) + filler_bytes)
    for entry in self.entries[:]:
      if entry.parent_id == 0xffff:
        continue
      entry.location = (temp // 512) + self.layout.getint("dir_blk") + 1
      temp = self.sector_align(temp + entry.size)

  def img_nblks(self):
     blks = self.layout.getint("dir_blk") + self.layout.getint("dir_nblks")
     for entry in self.entries:
       blks += self.sector_align(entry.size) // 512
     return blks

  def dir_used_nblks(self):
     return (self.sector_align((len(self.entries) * 32)) // 512) + 1

  def hdr1(self):
    hdr = PVOSHeader1 (
             hdr1_2_blk=self.layout.getint('dir_blk') & 0x00ff,
             hdr1_2_cpy=self.layout.getint('dir_blk') & 0x00ff,
             hdr2_blk=self.layout.getint('dir_blk') & 0xff00,
             dword3C=self.layout.getint('flash_nblks') - self.layout.getint('dword38'),
             dword44=self.layout.getint('flash_nblks') - self.layout.getint('dword38'),
             img_nblks=self.img_nblks(), filler70=b"\xff"*398
          )
    for (name, value) in list(self.layout.items()):
       if hasattr(hdr, name):
           attr_type = type(getattr(hdr, name))
           if attr_type is int:
              setattr(hdr, name, int(value, 0))
           elif attr_type is bytes:
              setattr(hdr, name, value.encode('latin-1'))
    return hdr

  def hdr2(self):
    hdr = PVOSHeader2 (filler34=b"\xff"*454)
    for (name, value) in list(self.layout.items()):
       if hasattr(hdr, name):
           attr_type = type(getattr(hdr, name))
           if attr_type is int:
              setattr(hdr, name, int(value, 0))
           elif attr_type is bytes:
              setattr(hdr, name, value.encode('latin-1'))
    return hdr

  def sector_align(self, sz):
    return sz + (-sz & 511)

  def write_nand(self, filename):
    hdr1 = self.hdr1()
    hdr2 = self.hdr2()
    f = open(filename, "w+b")
    f.write(hdr1.pack())
    f.write(b"\xff" * ((hdr1.hdr1_2_blk - 1) * 512))
    f.write(hdr1.pack())
    f.write(b"\xff" * ((hdr1.hdr2_blk - hdr1.hdr1_2_blk - 1) * 512))
    f.seek(512, 1)
    f.write(b"\xff" * ((hdr1.dir_blk - hdr1.hdr2_blk - 2) * 512))
    f.write(b"\xff" * 496)
    f.write(self.layout.get('valid').encode('ascii'))
    f.write(b"\x00" * 2)
    f.write(b"\xff" * 510)
    for entry in self.entries:
      f.write(entry)
      if entry.parent_id != 0xffff:
        off = f.tell()
        f.seek(entry.location * 512)
        filename = os.path.join(self.path, self.entries[entry.parent_id].name.decode('ascii'), entry.name.decode('ascii'))
        print("Adding %s at 0x%x (0x%x)" % (filename, entry.location * 512, entry.size))
        contents = open(filename, "rb").read()
        f.write(contents)
        f.write(b"\xff" * (self.sector_align(f.tell()) - f.tell()))
        f.seek(off)
    off = f.tell();
    filler_bytes = (self.layout.getint('dir_nblks') - self.dir_used_nblks()) * 512;
    f.write(b"\xff" * (self.sector_align(off + filler_bytes) - off))
    f.seek((hdr1.hdr2_blk + 1) * 512 ,0)
    print("Calculating NAND checksum...")
    for x in range (self.img_nblks() - (hdr1.hdr2_blk + 1)):
    	hdr2.datachksum.update(f.read(512))
    f.seek(hdr1.hdr2_blk * 512 ,0)
    f.write(hdr2.pack(self.timestamp))
