# Casio dictionary NAND dumper (python2)
#
# For Use with NAND dumps produced via TEST MENU > DATAOUT only.
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.

import struct, ctypes, sys, os, os.path, array
from optparse import OptionParser
from pprint import pprint

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

info = [
  {
    'hdr1blk': 0x100,
    'sig': 'CASIOPVOS400',
    'tag': 'PVOS20-VALIDAREA',
    'blk2off': lambda x: (x  * 528)
  },
  {
    'hdr1blk': 0x200,
    'sig': '\xffASIOPVOS500',
    'tag': 'PVOS20-VALIDAREA',
    'blk2off': lambda x: (x  * 528)
  },
  {
    'hdr1blk': 0x400,
    'sig': 'CASIOPVOS600',
    'tag': 'PVOS60-VALIDAREA',
    'blk2off': lambda x: (((x / 8) * 4314) + ((x % 8) * 539))
  },
  {
    'hdr1blk': 0x800,
    'sig': 'CASIOPVOS600',
    'tag': 'PVOS60-VALIDAREA',
    'blk2off': lambda x: (((x / 16) * 8628) + ((x % 16) * 539))
  },
]

class PVDirEntry(ctypes.BigEndianStructure):
  _fields_ = [
    ("filename",  char*16),    # name
    ("dir_id",    uint16_t),   # directory id
    ("parent",    uint16_t),   # containing directory id (for files)
    ("block_no",  uint32_t),   # starting block number
    ("size",      uint32_t),   # file size
    ("flags",     uint32_t),   #
  ]

class PVOSHeader1(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIOPVOS200
    ("magic",     uint16_t),  # 0x55AA
    ("blocksize", uint16_t),  # 0x200
    ("flash_nblks", uint32_t), #
    ("update_nblks",uint32_t), #
    ("dword18",   uint32_t), #  Zero?
    ("hdr1_2_blk",uint32_t), # blk no of second copy of hdr1??
    ("dir_blk",   uint32_t), # start blk of directory
    ("dir_nblks", uint32_t), # blk length of directory
    ("hdr2_blk",  uint32_t), #
    ("hdr1_2_cpy",uint32_t), # blk no of second copy of hdr1??
    ("dword30",   uint32_t), #
    ("dword34",   uint32_t), #
    ("dword38",   uint32_t), #
    ("dword3C",   uint32_t), #
    ("dword40",   uint32_t), #
    ("dword44",   uint32_t), #
    ("model",     char * 4), # 
    ("dword4C",   uint32_t), #
    ("img_nblks", uint32_t), #
    ("dword54",   uint32_t), #
    ("dword58",   uint32_t), #
    ("dword5C",   uint32_t), #
    ("dword60",   uint32_t), #
    ("dword64",   uint32_t), #
    ("ext_model", char*8),
    ("filler70",  char*398), #
    ("checksum",  uint16_t), #
   ]



def read_struct(f, struct):
    s = struct()
    slen = ctypes.sizeof(s)
    if isinstance(f, str):
      bytes = f
    else:
      bytes = f.read(slen)
    fit = min(len(bytes), slen)
    ctypes.memmove(ctypes.addressof(s), bytes, fit)
    return s

def print_line_or_lines(results, indent):
    """short values on same line, multi-line on later ones..."""
    if len(results) == 1:
        print results[0]
    else:
        print
        for result in results:
            print indent + result

def ctypes_pprint(cstruct, indent=""):
    """pretty print a ctypes Structure or Union"""
    
    for field_name, field_ctype in cstruct._fields_:
        field_value = getattr(cstruct, field_name)
        print indent + field_name,
        next_indent = indent + "    "
        pprint_name = "pprint_%s" % field_name
        pformat_name = "pformat_%s" % field_name
        if hasattr(cstruct, pprint_name):
            # no longer used
            getattr(cstruct, pprint_name)(next_indent)
        elif hasattr(cstruct, pformat_name):
            # counted-array and other common cases
            print_line_or_lines(getattr(cstruct, pformat_name)(), next_indent)
        elif hasattr(field_value, "pformat"):
            # common for small compound types
            print_line_or_lines(field_value.pformat(), next_indent)
        elif hasattr(field_value, "pprint"):
            # useful for Union selectors
            field_value.pprint(next_indent)
        elif hasattr(field_value, "_fields_"):
            # generic recursion
            print
            ctypes_pprint(field_value, next_indent)
        else:
            # generic simple (or unknown/uninteresting) value
            try:
              print "0x%X" % field_value
            except:
              print field_value



class nand:
  def __init__(self, f, info):
    self.hdr1blk = info['hdr1blk']
    self.tag = info['tag']
    self.blk2off = info['blk2off']
    self.file = f
    f.seek(0, 2)
    self.filesz = f.tell()
    f.seek(0, 0)
    blkno = self.hdr1blk
    while 1:
       blk = self.read_blk(blkno)
       if blk != None:
         tag = blk[496:]
         if tag == self.tag:
           self.fs_start = blkno + 2
           break
       blkno += 1

  def dir_blk(self):
    f.seek(self.blk2off(self.hdr1blk), 0)
    hdr1 = read_struct(f, PVOSHeader1)
    return hdr1.dir_blk

  def read_direntry(self, entryno):
    blkno = (entryno * 32) / 512
    offset = (entryno * 32) % 512
    bytes = self.read_blk(self.fs_start + blkno)
    entry = read_struct(bytes[offset:], PVDirEntry)
    if entry.dir_id == 0xffff:
      entry.block_no = entry.block_no - (self.dir_blk() + 1) + self.fs_start
    return entry
  
  def read_blk(self, blkno):
    off = self.blk2off(blkno)
    if off >= self.filesz:
       return None
    f.seek(off, 0)
    return f.read(512)
    

def dump_nand(n, extract):
  dirno = 0;
  de = n.read_direntry(dirno)
  while de.dir_id != 0xffff:
    dirno += 1
    print "Directory %r" % de.filename
    if extract:
      os.makedirs(de.filename)
    blkno = de.block_no
    fe = n.read_direntry(blkno)
    while de.dir_id == fe.parent:
      blkno += 1
      if extract:
        path = os.path.join(de.filename, fe.filename)
        f = open(path, "wb")
        offset = 0
        while offset < fe.size:
          bno = offset / 512
          read = min(512, fe.size - offset)
          bytes = n.read_blk(fe.block_no + bno)
          if bytes == None:
            print "    failed to copy file %s" % path
            break
          if read < 512:
            bytes = bytes[:read]
          f.write(bytes)
          offset += read
      print "  File %r" % fe.filename
      fe = n.read_direntry(blkno)
    de = n.read_direntry(dirno)
  
def find_nand(f):
  for desc in info:
    off = desc['blk2off'](desc['hdr1blk'])
    f.seek(off, 0)
    hdr1 = read_struct(f, PVOSHeader1)
    if hdr1.signature == desc['sig']:
      print "NAND Signature: %r" % hdr1.signature
      ctypes_pprint(hdr1)
      return nand(f, desc)
  return None

parser = OptionParser(usage="usage: %prog [options] filename")
parser.add_option('-e', '--extract', dest="extract", action='store_true', help='extract the bundled files', default=False)

(options, args) = parser.parse_args()
if len(args) != 1:
  parser.print_help()
  sys.exit(1)

f = open(args[0], "rb")
n = find_nand(f)
if n == None:
  print "Not valid NAND dump"
else:
  dump_nand(n, options.extract)
