import ctypes

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

class CheckSum16(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint16_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class CheckSum32(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint32_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class PVOSHeader1(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIOPVOS???
    ("magic",     uint16_t),  # 0x55AA
    ("blocksize", uint16_t),  # 0x200
    ("flash_nblks", uint32_t), #
    ("update_nblks",uint32_t), #
    ("dword18",   uint32_t), #  Zero?
    ("hdr1_2_blk",uint32_t), # blk no of second copy of hdr1??
    ("dir_blk",   uint32_t), # start blk of directory
    ("dir_nblks", uint32_t), # blk length of directory
    ("hdr2_blk",  uint32_t), # blk of second hdr
    ("hdr1_2_cpy",uint32_t), # blk no of second copy of hdr1?? always the same as hdr1_2_blk
    ("dword30",   uint32_t), # always 0xffffffff
    ("dword34",   uint32_t), # always 0xffffffff
    ("dword38",   uint32_t), #
    ("dword3C",   uint32_t), #
    ("dword40",   uint32_t), # always 0x0?
    ("dword44",   uint32_t), #
    ("model",     char * 4), # 
    ("dword4C",   uint32_t), #
    ("img_len",   uint32_t), # length of OS update image file in blocks
    ("dword54",   uint32_t), #
    ("dword58",   uint32_t), #
    ("dword5C",   uint32_t), #
    ("dword60",   uint32_t), #
    ("dword64",   uint32_t), #
    ("ext_model", char * 8), # extended model string
    ("filler70",  uint8_t*398), #
    ("checksum",  uint16_t), #
   ]

   def blk2off(self, blk):
     return blk * self.blocksize

class PVDateTime(ctypes.BigEndianStructure):
    _fields_ = [ ("bytes", uint8_t*8) ]
    def __str__(self):
        vals = tuple(self.bytes[i]&0xFF for i in range(7))
        return "%02X%02X/%02X/%02X %02X:%02X:%02X" % vals
    def pformat(self):
        return [self.__str__()]

class PVOSHeader2(ctypes.BigEndianStructure):
  _fields_ = [
    ("signature", char * 12), # CASIOPVOS???
    ("dwordC",    uint32_t), # All Zeros
    ("dword10",   uint32_t), # All Zeros
    ("dword14",   uint32_t), # All FF
    ("model",     char * 4), # model string
    ("datetime",  PVDateTime), # date time (bcd encoded)
    ("version",   uint16_t), # version number (bcd encoded)
    ("pad26",     uint16_t), # padding should be zero
    ("dword28",   uint32_t), # All Zeros
    ("datachksum",uint32_t), # NAND Checksum
    ("ext_model", char * 8), # ext model string
    ("filler2C",  uint8_t*454), #
    ("checksum",  uint16_t), #
  ]

class PVFilename(ctypes.BigEndianStructure):
    _fields_ = [ ("chars", uint16_t*16) ]
    def tostr(self):
        vals = [chr(self.chars[i]&0xFF) for i in range(16)]
        return "".join(vals).rstrip('\x00')
    def pformat(self):
        return [self.tostr()]

class PVDirEntry(ctypes.BigEndianStructure):
  _fields_ = [
    ("dir_id",    uint16_t),   # directory id
    ("parent",    uint16_t),   # containing directory id (for files)
    ("filename",  PVFilename), # utf-16be name
    ("block_no",  uint32_t),   # starting block number
    ("res_blks",  uint32_t),   # total blocks reserved for the file
    ("size",      uint32_t),   # tctual file size in bytes
    ("datetime",  PVDateTime), # date and time stamp
    ("flags",     uint32_t),   #
    ("dword3C",   uint32_t),   #
  ]
             
class PVDirEntryNew(ctypes.BigEndianStructure):
  _fields_ = [
    ("filename",  char*16),    # name
    ("dir_id",    uint16_t),   # directory id
    ("parent",    uint16_t),   # containing directory id (for files)
    ("block_no",  uint32_t),   # starting block number
    ("size",      uint32_t),   # file size
    ("flags",     uint32_t),   #
  ]

class CasioDicHeaderR(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03R
    ("model",     char * 4),  # L865
    ("magic",     uint32_t),  # 0x55AAAA55
    ("flash_size",uint32_t),  # total system flash size?
    ("block_size",uint32_t),  # flash block size?
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("dword20",   uint32_t),  #
    ("b_offset",  uint32_t),  #
    ("dword28",   uint32_t),  #
    ("dword2C",   uint32_t),  #
    ("boot_cksum",uint32_t),  # checksum of start data until this header
    ("extmodel",  char * 8),  # CY101
    ("filler3C",  uint8_t*66), #
    ("checksum",  uint16_t),  # header checksum
   ]

class ExtModelStr(ctypes.BigEndianStructure):
    _fields_ = [ ("chars", char*46) ]
    def __str__(self):
        slen = self.chars[0]
        if slen == 0xFF:
          return ""
        return (self.chars[1:slen+1]).decode('ascii', 'ignore')
    def pformat(self):
        return [self.__str__()]

class CasioDicHeaderB(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("a_offset",  uint32_t),  # offset of the A header
    ("entrypoint",uint32_t),  # main code entry point
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("farea_off", uint32_t),  #
    ("farea_len", uint32_t),  #
    ("datetime1", PVDateTime),#
    ("datetime2", PVDateTime),#
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("datasum1",  uint32_t),  #
    ("datasum2",  uint32_t),  #
    ("rom_cksum", uint32_t),  #
    ("filler44",  uint8_t*12),#
    ("extmodel",  ExtModelStr), # CY101
    ("checksum",  uint16_t),  # header checksum
   ]

class CasioDicHeaderA(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("name",      char *32),  #
    ("datetime",  PVDateTime),#
    ("version",   uint16_t),  #
    ("pad3A",     uint8_t*2), #
    ("entrypoint",uint32_t),  #
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("datasum",   uint32_t),  #
    ("filler48",  uint8_t*8),#
    ("extmodel",  char * 8),  # CY101
    ("filler5C",  uint8_t*38),#
    ("checksum",  uint16_t),  # header checksum
   ]

class DicsDirEntry(ctypes.BigEndianStructure):
  _fields_ = [
    ("name",      char*16),    # name
    ("dir_id",    uint16_t),   # directory id
    ("parent",    uint16_t),   # containing directory id (for files)
    ("offset",    uint32_t),   # file data offset
    ("size",      uint32_t),   # file size
    ("flags",     uint32_t),   #
  ]        

class DicsDirEntryOld(ctypes.BigEndianStructure):
  _fields_ = [
    ("dir_id",    uint16_t),   # directory id
    ("parent",    uint16_t),   # containing directory id (for files)
    ("name",      char*32),    # name
    ("offset",    uint32_t),   # file data offset
    ("unk28",     uint32_t),   #
    ("size",      uint32_t),   # file size
    ("datetime",  PVDateTime), #
    ("flags",     uint32_t),   #
    ("filler",  uint8_t*52),   #
  ]        

class DicsPreloadFile(ctypes.BigEndianStructure):
  _fields_ = [
    ("filename",  char*28),    # name
    ("idx",       uint32_t),   #
  ]        

class DicsFareaHeader(ctypes.BigEndianStructure):
  _fields_ = [
    ("count",  uint16_t),    # number of preload files
    ("filler", uint8_t*14),  #
  ]
