# Casio dictionary NOR patcher
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.



import struct, ctypes, sys, os, array
from optparse import OptionParser

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

patch_info = [
  {
    'checksum' : 0xae9afce,
    'offset' : 0x7f428,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  {
    'checksum' : 0xae9ff05,
    'offset' : 0x7f340,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  {
    'checksum' : 0x90d3e9b,
    'offset' : 0x7f914,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  {
    'checksum' : 0x90e5116,
    'offset' : 0x7fb34,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  {
    'checksum' : 0xae891ee,
    'offset' : 0x7ecfa,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  { #XD-D9800
    'checksum' : 0xa78438c,
    'offset' : 0x9216c,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  { #E-D800
    'checksum' : 0xa793d30,
    'offset' : 0x92194,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  { #XD-D1000
    'checksum' : 0x9a9f689,
    'offset' : 0x8f4c4,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
  { #XD-B9800
    'checksum' : 0x90d3ead,
    'offset' : 0x7f914,
    'original' : array.array('B', [0xed, 0x01]),
    'new' : array.array('B', [0xed, 0x00]),
  },
]

class CheckSum16(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint16_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class CheckSum32(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint32_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class PVDateTime(ctypes.BigEndianStructure):
    _fields_ = [ ("bytes", uint8_t*8) ]
    def __str__(self):
        vals = tuple(self.bytes[i]&0xFF for i in range(7))
        return "%02X%02X/%02X/%02X %02X:%02X:%02X" % vals
    def pformat(self):
        return [self.__str__()]

class ExtModelStr(ctypes.BigEndianStructure):
    _fields_ = [ ("chars", char*46) ]
    def __str__(self):
        slen = self.chars[0]
        if slen == 0xFF:
          return ""
        return (self.chars[1:slen+1])
    def pformat(self):
        return [self.__str__()]

class CasioDicHeaderR(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03R
    ("model",     char * 4),  # L865
    ("magic",     uint32_t),  # 0x55AAAA55
    ("dword14",   uint32_t),  # total system flash size?
    ("dword18",   uint32_t),  # flash block size?
    ("dword1C",   uint32_t),  #
    ("dword20",   uint32_t),  #
    ("b_offset",  uint32_t),  #
    ("dword28",   uint32_t),  #
    ("dword2C",   uint32_t),  #
    ("boot_cksum",uint32_t),  # checksum of start data until this header
    ("extmodel",  char * 8),  # CY101
    ("filler3C",  uint8_t*66), #
    ("checksum",  uint16_t),  # header checksum
   ]

class CasioDicHeaderB(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("a_offset",  uint32_t),  # offset of the A header
    ("entrypoint",uint32_t),  # main code entry point
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("farea_off", uint32_t),  #
    ("farea_len", uint32_t),  #
    ("datetime1", PVDateTime),#
    ("datetime2", PVDateTime),#
    ("version",   uint16_t),  #
    ("typecode",  uint8_t),   #
    ("unk37",     uint8_t),   #
    ("datasum1",  uint32_t),  #
    ("datasum2",  uint32_t),  #
    ("rom_cksum", uint32_t),  #
    ("filler44",  uint8_t*12),#
    ("extmodel",  ExtModelStr), # CY101
    ("checksum",  uint16_t),  # header checksum
   ]

def read_struct(f, struct):
    s = struct()
    slen = ctypes.sizeof(s)
    if isinstance(f, bytes):
      byte_data = f
    else:
      byte_data = f.read(slen)
    fit = min(len(byte_data), slen)
    ctypes.memmove(ctypes.addressof(s), byte_data, fit)
    return s


def get_patch_info(chksum):
  for info in patch_info:
    if info['checksum'] == chksum:
      return info
  return None

def patch_nor(f):
  r_offset = f.tell()
  r_hdr = read_struct(f, CasioDicHeaderR)
  f.seek(r_hdr.b_offset, 0)
  b_hdr = read_struct(f, CasioDicHeaderB)
  f.seek(0, 2)
  filesize = f.tell()
  pinfo = get_patch_info(b_hdr.datasum1)
  if pinfo == None:
    print("Can't find patch info for ROM with chksum of 0x%x" % b_hdr.datasum1)
  else:
     print("Patching %s NOR..." % r_hdr.model)
     f.seek(pinfo['offset'], 0)
     bytes_read = f.read(len(pinfo['original']))
     if bytes_read != pinfo['original'].tobytes():
       print("Wrong data at offset 0x%x" % pinfo['offset'])
     else:
       mod = 0x500
       if r_offset == 0xff80:
         mod = 0x300
       datasum1 = CheckSum32()
       rom_cksum = CheckSum32()
       checksum = CheckSum16()
       f.seek(pinfo['offset'], 0)
       f.write(pinfo['new'].tobytes())
       f.seek(b_hdr.entrypoint & 0x7fffffff, 0)
       datasum1.update(f.read(b_hdr.datalen))
       b_hdr.datasum1 = datasum1.chksum
       f.seek(0)
       rom_cksum.update(f.read(r_offset - mod))
       rom_cksum.update(b"\xff" * mod)
       f.seek(r_offset, 0)
       rom_cksum.update(f.read(r_hdr.b_offset - f.tell()))
       f.seek(r_hdr.b_offset + 0x80, 0)
       rom_cksum.update(f.read(filesize - f.tell()))
       if filesize < r_hdr.dword14:
         bytes_fill = b"\xff" * (r_hdr.dword14 - filesize)
         rom_cksum.update(bytes_fill)
       b_hdr.rom_cksum = rom_cksum.chksum
       checksum.update(bytes(b_hdr)[:-2])
       b_hdr.checksum = checksum.chksum
       f.seek(r_hdr.b_offset, 0)
       print("Patching checksums...")
       f.write(b_hdr)

def get_dics(f):
  for offs in [0xFF80, 0x1FF80]:
    f.seek(offs)
    sig = f.read(9)
    if sig == b"CASIODICS":
      f.seek(-9, 1)
      return f
  return None

parser = OptionParser(usage="usage: %prog input")

(options, args) = parser.parse_args()
if len(args) != 1:
  parser.print_help()
  sys.exit(1)

f = open(args[0], "r+b")
f = get_dics(f)
if f != None:
  patch_nor(f)
else:
  print("Not a NOR image")