# Casio dictionary NOR Checksum verifier
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.


import struct, ctypes, sys, os, array
from optparse import OptionParser

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

class CheckSum16(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint16_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class CheckSum32(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint32_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       self.chksum += sum(data)

class PVDateTime(ctypes.BigEndianStructure):
    _fields_ = [ ("bytes", uint8_t*8) ]
    def __str__(self):
        vals = tuple(self.bytes[i]&0xFF for i in range(7))
        return "%02X%02X/%02X/%02X %02X:%02X:%02X" % vals
    def pformat(self):
        return [self.__str__()]

class ExtModelStr(ctypes.BigEndianStructure):
    _fields_ = [ ("chars", char*46) ]
    def __str__(self):
        slen = self.chars[0]
        if slen == 0xFF:
          return ""
        return (self.chars[1:slen+1])
    def pformat(self):
        return [self.__str__()]
             
def read_struct(f, struct):
    s = struct()
    slen = ctypes.sizeof(s)
    if isinstance(f, bytes):
      byte_data = f
    else:
      byte_data = f.read(slen)
    fit = min(len(byte_data), slen)
    ctypes.memmove(ctypes.addressof(s), byte_data, fit)
    return s

class CasioDicHeaderR(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03R
    ("model",     char * 4),  # L865
    ("magic",     uint32_t),  # 0x55AAAA55
    ("flash_size",uint32_t),  # total system flash size?
    ("block_size",uint32_t),  # flash block size?
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("dword20",   uint32_t),  #
    ("b_offset",  uint32_t),  #
    ("dword28",   uint32_t),  #
    ("dword2C",   uint32_t),  #
    ("boot_cksum",uint32_t),  # checksum of start data until this header
    ("extmodel",  char * 8),  # CY101
    ("filler3C",  uint8_t*66), #
    ("checksum",  uint16_t),  # header checksum
   ]

class CasioDicHeaderB(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("a_offset",  uint32_t),  # offset of the A header
    ("entrypoint",uint32_t),  # main code entry point
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("farea_off", uint32_t),  #
    ("farea_len", uint32_t),  #
    ("datetime1", PVDateTime),#
    ("datetime2", PVDateTime),#
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("datasum1",  uint32_t),  #
    ("datasum2",  uint32_t),  #
    ("rom_cksum", uint32_t),  #
    ("filler44",  uint8_t*12),#
    ("extmodel",  ExtModelStr), # CY101
    ("checksum",  uint16_t),  # header checksum
   ]

class CasioDicHeaderA(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIODICS03B
    ("model",     char * 4),  # L865
    ("name",      char * 32), #
    ("datetime",  PVDateTime),#
    ("version",   uint16_t),  #
    ("padding",   uint8_t*2), #
    ("entrypoint",uint32_t),  #
    ("datalen",   uint32_t),  # length of data covered by checksum
    ("datasum",   uint32_t),  #
    ("filler48",  uint8_t*8), #
    ("extmodel",  char * 8),  # CY101
    ("filler5C",  uint8_t*38),#
    ("checksum",  uint16_t),  # header checksum
   ]

def get_tail(f, r_offset):
  f.seek(r_offset, 0)
  hdr_r = read_struct(f, CasioDicHeaderR)
  f.seek(0,2)
  filesize = f.tell()
  f.seek(filesize - 0x05, 0)
  thetail = f.read(0x05)
  f.seek(filesize - 0x10, 0)
  if ((filesize < hdr_r.flash_size) and (f.read(0x05) != thetail)):
    tail = filesize
  else:
    i = 6
    f.seek(filesize - i, 0)
    while (f.read(0x05) == thetail):
      f.seek(filesize - i,0)
      i= i + 5
    tail=f.tell()
  if r_offset == 0xff80:
    tail = tail + (-tail & 0xfff)
  else:
    tail = tail + (-tail & 0xf)
  return tail

def calc_chksums(f):
  mod = 0x500
  boot_cksum = CheckSum32()
  datasum1 = CheckSum32()
  datasum2 = CheckSum32()
  datasum = CheckSum32()
  rom_cksum = CheckSum32()
  hdr_r_cksum = CheckSum16()
  hdr_b_cksum = CheckSum16()
  hdr_a_cksum = CheckSum16()
  r_offset = f.tell()
  if r_offset == 0xff80:
    mod = 0x300
  f.seek(0, 2)
  filesize = f.tell()
  f.seek(r_offset, 0)
  hdr_r = read_struct(f, CasioDicHeaderR)
  hdr_r_cksum.update(bytes(hdr_r)[:-2])
  f.seek(hdr_r.b_offset, 0)
  hdr_b = read_struct(f, CasioDicHeaderB)
  hdr_b_cksum.update(bytes(hdr_b)[:-2])
  f.seek(hdr_b.a_offset, 0)
  hdr_a = read_struct(f, CasioDicHeaderA)
  hdr_a_cksum.update(bytes(hdr_a)[:-2])
  f.seek(0, 0)
  boot_cksum.update(f.read(r_offset - mod))
  boot_cksum.update(b"\xff" * mod)
  if r_offset == 0xff80:
    f.seek(r_offset + 0x80)
    boot_cksum.update(f.read(hdr_r.b_offset - f.tell() - 0xa0)); #Is this correct?
  f.seek(hdr_b.entrypoint & 0x7fffffff, 0)
  datasum1.update(f.read(hdr_b.datalen))
  tail = get_tail(f, r_offset)
  f.seek(hdr_b.datalen + hdr_b.farea_off, 0)
  datasum2len = tail - hdr_b.datalen - hdr_b.farea_off
  datasum2.update(f.read(datasum2len))
  if tail > filesize:
    datasum2.update(b"\xff" * (tail - filesize))
  f.seek(hdr_a.entrypoint & 0x7fffffff, 0)
  datasum.update(f.read(hdr_a.datalen))
  f.seek(0)
  rom_cksum.update(f.read(r_offset - mod))
  rom_cksum.update(b"\xff" * mod)
  f.seek(r_offset, 0)
  rom_cksum.update(f.read(hdr_r.b_offset - f.tell()))
  f.seek(hdr_r.b_offset + 0x80, 0)
  rom_cksum.update(f.read(filesize - f.tell()))
  if filesize < hdr_r.flash_size:
    padding_bytes = b"\xff" * (hdr_r.flash_size - filesize)
    rom_cksum.update(padding_bytes)
  print("Model %r" % hdr_r.model)
  print("===R Header===")
  print("hdr_r.boot_cksum: %x (%x) %s" % (hdr_r.boot_cksum, boot_cksum.chksum, "OK" if hdr_r.boot_cksum == boot_cksum.chksum else "Fail"))
  print("hdr_r.checksum: %x (%x) %s" % (hdr_r.checksum, hdr_r_cksum.chksum, "OK" if hdr_r.checksum == hdr_r_cksum.chksum else "Fail"))
  print("===B Header===")
  print("hdr_b.datasum1: %x (%x) %s" % (hdr_b.datasum1, datasum1.chksum, "OK" if hdr_b.datasum1 == datasum1.chksum else "Fail"))
  print("hdr_b.datasum2: %x (%x) %s" % (hdr_b.datasum2, datasum2.chksum, "OK" if hdr_b.datasum2 == datasum2.chksum else "Fail"))
  print("hdr_b.rom_cksum: %x (%x) %s" % (hdr_b.rom_cksum, rom_cksum.chksum, "OK" if hdr_b.rom_cksum == rom_cksum.chksum else "Fail"))
  print("hdr_b.checksum: %x (%x) %s" % (hdr_b.checksum, hdr_b_cksum.chksum, "OK" if hdr_b.checksum == hdr_b_cksum.chksum else "Fail"))
  print("===A Header===")
  print("hdr_a.datasum: %x (%x) %s" % (hdr_a.datasum, datasum.chksum, "OK" if hdr_a.datasum == datasum.chksum else "Fail"))
  print("hdr_a.checksum: %x (%x) %s" % (hdr_a.checksum, hdr_a_cksum.chksum, "OK" if hdr_a.checksum == hdr_a_cksum.chksum else "Fail"))

def get_dics(f):
  for offs in [0xFF80, 0x1FF80]:
    f.seek(offs)
    sig = f.read(9)
    if sig == b"CASIODICS":
      f.seek(-9, 1)
      return f
  return None

parser = OptionParser(usage="usage: %prog filename")

(options, args) = parser.parse_args()
if len(args) != 1:
  parser.print_help()
  sys.exit(1)

print("Parsing file %s" % args[0])
f = open(args[0], "rb")
f2 = get_dics(f)
if f2 != None:
  calc_chksums(f2)
else:
  print("not a NOR image")