import sys, os, ctypes, struct, array, io
from optparse import OptionParser
from pathlib import Path
from structs import *

def read_struct(f, struct_class):
    s = struct_class()
    slen = ctypes.sizeof(s)
    
    if isinstance(f, bytes):
        byte_data = f
    else:
        byte_data = f.read(slen)
        
    fit = min(len(byte_data), slen)
    ctypes.memmove(ctypes.addressof(s), byte_data, fit)
    return s

def parse_farea(farea_data, etype):
    f = io.BytesIO(farea_data)
    fhdr = read_struct(f, DicsFareaHeader)
    
    preload_files = []
    if fhdr.count != 0:
        for _ in range(fhdr.count):
            preload_files.append(read_struct(f, DicsPreloadFile))

    dirs = {}
    files = []
    all_entries = []
    
    elen = ctypes.sizeof(etype())
    dummy = b'\xff' * elen
    
    while f.tell() < len(farea_data):
        pos = f.tell()
        s = f.read(elen)
        if not s or len(s) < elen:
            break
        
        if s == dummy:
            continue
        
        f.seek(pos)
        entry = read_struct(f, etype)
        all_entries.append(entry)

        if entry.dir_id != 0xFFFF:
            dirs[entry.dir_id] = entry
        else:
            files.append(entry)
            
    return fhdr, preload_files, dirs, files, all_entries

def build_farea(fhdr, preload_files, entries):
    farea_bytes = bytearray()
    farea_bytes.extend(bytes(fhdr))
    for pf in preload_files:
        farea_bytes.extend(bytes(pf))
    
    for entry in entries:
        farea_bytes.extend(bytes(entry))
        
    return farea_bytes

def get_tail_from_data(data, r_offset, flash_size):
    filesize = len(data)
    the_tail_bytes = data[filesize - 5 : filesize]
    
    if filesize < flash_size and data[filesize - 0x10 : filesize - 0x10 + 5] != the_tail_bytes:
        tail = filesize
    else:
        i = 6
        while (filesize - i + 5) > 0 and data[filesize - i : filesize - i + 5] == the_tail_bytes:
            i += 5
        tail = filesize - i + 5

    if r_offset == 0xff80:
        tail = tail + (-tail & 0xfff)
    else:
        tail = tail + (-tail & 0xf)
    return tail

def update_checksums(r_offset, hdr_r, hdr_b, hdr_a, full_data):
    mod = 0x500 if r_offset != 0xff80 else 0x300
    filesize = len(full_data)

    # boot_cksum
    boot_cksum = CheckSum32()
    boot_cksum.update(full_data[0 : r_offset - mod])
    boot_cksum.update(b"\xff" * mod)
    if r_offset == 0xff80:
        boot_cksum.update(full_data[r_offset + 0x80 : hdr_r.b_offset - 0xa0])
    hdr_r.boot_cksum = boot_cksum.chksum

    # datasum1
    datasum1 = CheckSum32()
    datasum1.update(full_data[hdr_b.entrypoint & 0x7fffffff : (hdr_b.entrypoint & 0x7fffffff) + hdr_b.datalen])
    hdr_b.datasum1 = datasum1.chksum

    # datasum2
    tail = get_tail_from_data(full_data, r_offset, hdr_r.flash_size)
    datasum2 = CheckSum32()
    datasum2_start = hdr_b.farea_off + hdr_b.datalen
    datasum2_len = tail - datasum2_start
    datasum2.update(full_data[datasum2_start : datasum2_start + datasum2_len])
    if tail > filesize:
        datasum2.update(b"\xff" * (tail - filesize))
    hdr_b.datasum2 = datasum2.chksum
    
    # datasum (A)
    datasum_a = CheckSum32()
    datasum_a.update(full_data[hdr_a.entrypoint & 0x7fffffff : (hdr_a.entrypoint & 0x7fffffff) + hdr_a.datalen])
    hdr_a.datasum = datasum_a.chksum

    # rom_cksum
    rom_cksum = CheckSum32()
    rom_cksum.update(full_data[0 : r_offset - mod])
    rom_cksum.update(b"\xff" * mod)
    rom_cksum.update(full_data[r_offset : hdr_r.b_offset])
    rom_cksum.update(full_data[hdr_r.b_offset + 0x80 : filesize])
    if filesize < hdr_r.flash_size:
        rom_cksum.update(b"\xff" * (hdr_r.flash_size - filesize))
    hdr_b.rom_cksum = rom_cksum.chksum

    # Header checksums (MUST BE LAST)
    hdr_r_cksum = CheckSum16()
    hdr_r_cksum.update(bytes(hdr_r)[:-2])
    hdr_r.checksum = hdr_r_cksum.chksum

    hdr_b_cksum = CheckSum16()
    hdr_b_cksum.update(bytes(hdr_b)[:-2])
    hdr_b.checksum = hdr_b_cksum.chksum

    hdr_a_cksum = CheckSum16()
    hdr_a_cksum.update(bytes(hdr_a)[:-2])
    hdr_a.checksum = hdr_a_cksum.chksum

def map_free_space(all_entries, manifest, flash_size):
    used_blocks = []
    # Add manifest blocks as used space first
    for name, attrs in manifest.items():
        # content_data is where files live, so we don't mark it as a single used block
        if name not in ['block_09_content_data.bin', 'block_10_final_tail.bin']:
             used_blocks.append((attrs['start'], attrs['end']))

    for entry in all_entries:
        if entry.size > 0:
            used_blocks.append((entry.offset, entry.offset + entry.size))
    
    used_blocks.sort()
    
    merged_blocks = []
    if not used_blocks:
        return [(0, flash_size)]

    current_start, current_end = used_blocks[0]
    for next_start, next_end in used_blocks[1:]:
        if next_start < current_end:
            current_end = max(current_end, next_end)
        else:
            merged_blocks.append((current_start, current_end))
            current_start, current_end = next_start, next_end
    merged_blocks.append((current_start, current_end))
    
    free_blocks = []
    last_end = 0
    for start, end in merged_blocks:
        if start > last_end:
            free_blocks.append((last_end, start - last_end))
        last_end = end
    
    if last_end < flash_size:
        free_blocks.append((last_end, flash_size - last_end))
        
    return sorted(free_blocks, key=lambda x: x[1], reverse=True)

def main():
    parser = OptionParser(usage="usage: %prog [options] <input_directory> <output_file>")
    parser.add_option("-r", "--allow-relocate", action="store_true", dest="relocate", default=False, help="Allow relocating/adding/deleting files.")
    (options, args) = parser.parse_args()

    if len(args) != 2:
        parser.error("Incorrect number of arguments")
        
    input_dir = args[0]
    output_file = args[1]

    print(f"Building firmware from '{input_dir}' into '{output_file}'")
    if options.relocate:
        print("Advanced mode enabled (relocate/add/delete).")

    metadata_dir = os.path.join(input_dir, "metadata")
    content_dir = os.path.join(input_dir, "content")

    blocks = {}
    manifest = {}
    with open(os.path.join(metadata_dir, "manifest.txt"), "r") as f:
        for line in f:
            name, _, rest = line.partition(":")
            parts = rest.strip().split(", ")
            attrs = {p.split("=")[0]: int(p.split("=")[1], 16) for p in parts}
            manifest[name] = attrs
            
            with open(os.path.join(metadata_dir, name), "rb") as bf:
                blocks[name] = bf.read()

    hdr_r = read_struct(blocks['block_02_header_R.bin'], CasioDicHeaderR)
    hdr_b = read_struct(blocks['block_04_header_B.bin'], CasioDicHeaderB)
    hdr_a = read_struct(blocks['block_06_header_A.bin'], CasioDicHeaderA)
    etype = DicsDirEntry if hdr_b.a_offset > 0x200000 else DicsDirEntryOld
    
    original_farea = blocks['block_08_farea.bin']
    fhdr, preload_files, dirs, files, all_original_entries = parse_farea(original_farea, etype)

    content_dir_path = Path(content_dir)
    content_files = {}
    for full_path in content_dir_path.rglob('*'):
        if full_path.is_file():
            rel_path = full_path.relative_to(content_dir_path).as_posix()
            content_files[rel_path] = str(full_path)
    
    final_data = bytearray()
    for name in sorted(manifest.keys()):
        final_data.extend(blocks[name])

    new_farea_entries = []
    processed_content_files = set()

    if options.relocate:
        free_space = map_free_space(all_original_entries, manifest, hdr_r.flash_size)

    # Deletion/Modification pass
    for entry in all_original_entries:
        if entry.dir_id != 0xFFFF: # Is a directory
            new_farea_entries.append(entry)
            continue

        path_parts = [entry.name.decode('ascii', 'ignore').rstrip('\x00')]
        parent_id = entry.parent
        while parent_id != 0xFFFF:
            parent_entry = dirs.get(parent_id)
            if not parent_entry: sys.exit(f"Error: Broken parent link for {path_parts[0]}")
            path_parts.append(parent_entry.name.decode('ascii', 'ignore').rstrip('\x00'))
            parent_id = parent_entry.parent
        name_str = "/".join(reversed(path_parts))

        if name_str in content_files:
            processed_content_files.add(name_str)
            with open(content_files[name_str], 'rb') as f:
                new_data = f.read()
            
            if len(new_data) > entry.size and options.relocate:
                new_offset, found_gap_index = -1, -1
                for i, (start, size) in enumerate(free_space):
                    if size >= len(new_data):
                        new_offset, found_gap_index = start, i
                        break
                if new_offset == -1: sys.exit(f"Error: Not enough space for {name_str}")

                print(f"Relocating {name_str} from 0x{entry.offset:X} to 0x{new_offset:X}")
                _, old_gap_size = free_space.pop(found_gap_index)
                if old_gap_size > len(new_data):
                    free_space.append((new_offset + len(new_data), old_gap_size - len(new_data)))
                free_space.append((entry.offset, entry.size))
                free_space.sort(key=lambda x: x[1], reverse=True)
                entry.offset = new_offset
            elif len(new_data) > entry.size and not options.relocate:
                 sys.exit(f"Error: {name_str} is larger than original. Use --allow-relocate.")
            
            final_data[entry.offset : entry.offset + len(new_data)] = new_data
            if len(new_data) < entry.size:
                final_data[entry.offset + len(new_data) : entry.offset + entry.size] = b'\xff' * (entry.size - len(new_data))
            entry.size = len(new_data)
            new_farea_entries.append(entry)
        elif options.relocate:
            print(f"Deleting {name_str}")
            free_space.append((entry.offset, entry.size))
            free_space.sort(key=lambda x: x[1], reverse=True)
        else:
            new_farea_entries.append(entry)

    # Addition pass
    if options.relocate:
        new_files_to_add = set(content_files.keys()) - processed_content_files
        for name_str in new_files_to_add:
            print(f"Adding new file: {name_str}")
            path = Path(name_str)
            parent_path_str = str(path.parent)
            parent_id = -1
            if parent_path_str == '.':
                parent_id = 0xFFFF
            else:
                for dir_entry in dirs.values():
                    # This is a simplified parent lookup, might need improvement
                    if dir_entry.name.decode('ascii', 'ignore').rstrip('\x00') == parent_path_str:
                        parent_id = dir_entry.dir_id
                        break
            if parent_id == -1: sys.exit(f"Error: Could not find parent directory for {name_str}")

            with open(content_files[name_str], 'rb') as f:
                new_data = f.read()

            new_offset, found_gap_index = -1, -1
            for i, (start, size) in enumerate(free_space):
                if size >= len(new_data):
                    new_offset, found_gap_index = start, i
                    break
            if new_offset == -1: sys.exit(f"Error: Not enough space for new file {name_str}")

            _, old_gap_size = free_space.pop(found_gap_index)
            if old_gap_size > len(new_data):
                free_space.append((new_offset + len(new_data), old_gap_size - len(new_data)))
            free_space.sort(key=lambda x: x[1], reverse=True)

            new_entry = etype()
            new_entry.name = path.name.encode('ascii')
            new_entry.dir_id = 0xFFFF
            new_entry.parent = parent_id
            new_entry.offset = new_offset
            new_entry.size = len(new_data)
            new_entry.flags = 0x10C # Default flags, may need adjustment
            new_farea_entries.append(new_entry)
            final_data[new_offset : new_offset + len(new_data)] = new_data

    new_farea_bin = build_farea(fhdr, preload_files, new_farea_entries)
    if not options.relocate:
        original_farea_size = manifest['block_08_farea.bin']['size']
        if len(new_farea_bin) > original_farea_size:
            sys.exit(f"Error: New farea size is larger than original.")
        new_farea_bin.extend(b'\xff' * (original_farea_size - len(new_farea_bin)))
    
    farea_offset = manifest['block_08_farea.bin']['start']
    final_data[farea_offset : farea_offset + len(new_farea_bin)] = new_farea_bin
    if len(new_farea_bin) < manifest['block_08_farea.bin']['size']:
        fill_start = farea_offset + len(new_farea_bin)
        fill_end = manifest['block_08_farea.bin']['end']
        final_data[fill_start:fill_end] = b'\xff' * (fill_end - fill_start)

    hdr_b.farea_len = len(new_farea_bin)
    
    r_offset = manifest['block_02_header_R.bin']['start']
    update_checksums(r_offset, hdr_r, hdr_b, hdr_a, bytes(final_data))

    hdr_r_offset = manifest['block_02_header_R.bin']['start']
    hdr_b_offset = manifest['block_04_header_B.bin']['start']
    hdr_a_offset = manifest['block_06_header_A.bin']['start']

    final_data[hdr_r_offset:hdr_r_offset+ctypes.sizeof(hdr_r)] = bytes(hdr_r)
    final_data[hdr_b_offset:hdr_b_offset+ctypes.sizeof(hdr_b)] = bytes(hdr_b)
    final_data[hdr_a_offset:hdr_a_offset+ctypes.sizeof(hdr_a)] = bytes(hdr_a)

    with open(output_file, "wb") as f:
        f.write(final_data)
        
    print(f"Successfully built '{output_file}' ({len(final_data)} bytes)")

if __name__ == "__main__":
    main()
