# Casio dictionary NAND Checksum verifier
#
# This software is provided ‘as-is’, without any express or implied
# warranty. In no event will the authors be held liable for any damages
# arising from the use of this software.
# 
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
# 
# 1. The origin of this software must not be misrepresented; you must not
# claim that you wrote the original software. If you use this software
# in a product, an acknowledgment in the product documentation would be
# appreciated but is not required.
# 
# 2. Altered source versions must be plainly marked as such, and must not be
# misrepresented as being the original software.
# 
# 3. This notice may not be removed or altered from any source
# distribution.



import struct, ctypes, sys, os, array
from optparse import OptionParser

uint8_t  = ctypes.c_byte
uint16_t = ctypes.c_ushort
uint32_t = ctypes.c_uint
char     = ctypes.c_char
wchar_t  = ctypes.c_wchar

class CheckSum16(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint16_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       for i in range(len(data)):
         self.chksum += data[i]

class CheckSum32(ctypes.BigEndianStructure):
    _fields_ = [ ("chksum", uint32_t) ]
    def init(self):
        self.chksum = 0
    def update(self, data):
       for i in range(len(data)):
         self.chksum += data[i]

class PVDateTime(ctypes.BigEndianStructure):
    _fields_ = [ ("bytes", uint8_t*8) ]
    def __str__(self):
        vals = tuple(self.bytes[i]&0xFF for i in range(7))
        return "%02X%02X/%02X/%02X %02X:%02X:%02X" % vals
    def pformat(self):
        return [self.__str__()]
             
def read_struct(f, struct):
    s = struct()
    slen = ctypes.sizeof(s)
    if isinstance(f, bytes):
      byte_data = f
    else:
      byte_data = f.read(slen)
    fit = min(len(byte_data), slen)
    ctypes.memmove(ctypes.addressof(s), byte_data, fit)
    return s

class PVOSHeader1(ctypes.BigEndianStructure):
   _fields_ = [
    ("signature", char * 12), # CASIOPVOS???
    ("magic",     uint16_t),  # 0x55AA
    ("blocksize", uint16_t),  # 0x200
    ("flash_nblks", uint32_t), #
    ("update_nblks",uint32_t), #
    ("dword18",   uint32_t), #  Zero?
    ("hdr1_2_blk",uint32_t), # blk no of second copy of hdr1??
    ("dir_blk",   uint32_t), # start blk of directory
    ("dir_nblks", uint32_t), # blk length of directory
    ("hdr2_blk",  uint32_t), # blk of second hdr
    ("hdr1_2_cpy",uint32_t), # blk no of second copy of hdr1?? always the same as hdr1_2_blk
    ("dword30",   uint32_t), # always 0xffffffff
    ("dword34",   uint32_t), # always 0xffffffff
    ("dword38",   uint32_t), #
    ("dword3C",   uint32_t), #
    ("dword40",   uint32_t), # always 0x0?
    ("dword44",   uint32_t), #
    ("model",     char * 4), # 
    ("dword4C",   uint32_t), #
    ("img_len",   uint32_t), # length of OS update image file in blocks
    ("dword54",   uint32_t), #
    ("dword58",   uint32_t), #
    ("dword5C",   uint32_t), #
    ("dword60",   uint32_t), #
    ("dword64",   uint32_t), #
    ("ext_model", char * 8), # extended model string
    ("filler70",  uint8_t*398), #
    ("checksum",  uint16_t), #
   ]

class PVOSHeader2(ctypes.BigEndianStructure):
  _fields_ = [
    ("signature", char * 12), # CASIOPVOS???
    ("dwordC",    uint32_t), # All Zeros
    ("dword10",   uint32_t), # All Zeros
    ("dword14",   uint32_t), # All FF
    ("model",     char * 4), # model string
    ("datetime",  PVDateTime), # date time (bcd encoded)
    ("version",   uint16_t), # version number (bcd encoded)
    ("pad26",     uint16_t), # padding should be zero
    ("dword28",   uint32_t), # All Zeros
    ("datachksum",uint32_t), # NAND Checksum
    ("ext_model", char * 8), # ext model string
    ("filler2C",  uint8_t*454), #
    ("checksum",  uint16_t), #
  ]

def calc_chksums(f):
  nand_cksum = CheckSum32()
  hdr_1_cksum = CheckSum16()
  hdr_2_cksum = CheckSum16()
  hdr_1 = read_struct(f, PVOSHeader1)
  hdr_1_cksum.update(bytes(hdr_1)[:-2])
  f.seek(hdr_1.hdr2_blk * hdr_1.blocksize, 0)
  hdr_2 = read_struct(f, PVOSHeader2)
  hdr_2_cksum.update(bytes(hdr_2)[:-2])
  f.seek(0, 2)
  fileblks = f.tell() // hdr_1.blocksize;
  f.seek((hdr_1.hdr2_blk + 1) * hdr_1.blocksize, 0)
  for x in range (fileblks - (f.tell() // hdr_1.blocksize)):
    nand_cksum.update(f.read(512))
  print("Model %r" % hdr_1.model)
  print("===PVOS Header 1===")
  print("hdr_1.checksum: %x (%x) %s" % (hdr_1.checksum, hdr_1_cksum.chksum, "OK" if hdr_1.checksum == hdr_1_cksum.chksum else "Fail"))
  print("===PVOS Header 2===")
  print("hdr_2.datachksum: %x (%x) %s" % (hdr_2.datachksum, nand_cksum.chksum, "OK" if hdr_2.datachksum == nand_cksum.chksum else "Fail"))
  print("hdr_2.checksum: %x (%x) %s" % (hdr_2.checksum, hdr_2_cksum.chksum, "OK" if hdr_2.checksum == hdr_2_cksum.chksum else "Fail"))


def get_dics(f):
  hdr = read_struct(f, PVOSHeader1)
  sig = hdr.signature[:9]
  if sig == b"CASIOPVOS" and hdr.magic == 0x55aa:
    print("Signature: %s" % (hdr.signature))
    if hdr.magic == 0x55aa:
      f.seek(0,0)
      return f
  return None

parser = OptionParser(usage="usage: %prog filename")

(options, args) = parser.parse_args()
if len(args) != 1:
  parser.print_help()
  sys.exit(1)

print("Parsing file %s" % args[0])
f = open(args[0], "rb")
f2 = get_dics(f)
if f2 != None:
  calc_chksums(f2)
else:
  print("not a NAND image")